<?php
namespace App\Controllers;

use App\Core\View;
use App\Core\Response;
use App\Core\CSRF;
use App\Core\Auth;
use App\Core\Utils;
use App\Models\Company;
use App\Models\User;

class AuthController {
  public function showLogin(): void {
    View::render('auth/login', ['error'=>null]);
  }

  public function showCompanyRegister(): void {
    View::render('auth/register_company', ['error'=>null]);
  }

  public function registerCompany(): void {
    CSRF::verify();

    $name = trim($_POST['company_name'] ?? '');
    $email = trim($_POST['company_email'] ?? '');
    $adminName = trim($_POST['admin_name'] ?? '');
    $adminEmail = trim($_POST['admin_email'] ?? '');
    $adminPass = (string)($_POST['admin_password'] ?? '');

    if ($name==='' || $email==='' || $adminName==='' || $adminEmail==='' || strlen($adminPass) < 8) {
      View::render('auth/register_company', ['error'=>'Please fill all required fields. Admin password must be 8+ chars.']);
      return;
    }

    if (!filter_var($email, FILTER_VALIDATE_EMAIL) || !filter_var($adminEmail, FILTER_VALIDATE_EMAIL)) {
      View::render('auth/register_company', ['error'=>'Invalid email format.']);
      return;
    }

    if (Company::findByEmail($email)) {
      View::render('auth/register_company', ['error'=>'Company email already registered.']);
      return;
    }

    $companyId = Company::create([
      'name'=>$name,
      'email'=>$email,
      'phone'=>trim($_POST['company_phone'] ?? ''),
      'address'=>trim($_POST['company_address'] ?? ''),
    ]);

    $hash = password_hash($adminPass, PASSWORD_DEFAULT);
    $userId = User::create([
      'company_id'=>$companyId,
      'name'=>$adminName,
      'email'=>$adminEmail,
      'password_hash'=>$hash,
      'role'=>'company_admin'
    ]);

    // Auto-login admin
    Auth::login([
      'id'=>$userId,
      'company_id'=>$companyId,
      'name'=>$adminName,
      'email'=>$adminEmail,
      'role'=>'company_admin'
    ]);

    Response::redirect('/dashboard');
  }

  public function login(): void {
    CSRF::verify();

    $companyEmail = trim($_POST['company_email'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password = (string)($_POST['password'] ?? '');

    if ($companyEmail==='' || $email==='' || $password==='') {
      View::render('auth/login', ['error'=>'Please enter company email, user email, and password.']);
      return;
    }

    if (!filter_var($companyEmail, FILTER_VALIDATE_EMAIL) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
      View::render('auth/login', ['error'=>'Invalid email format.']);
      return;
    }

    $company = Company::findByEmail($companyEmail);
    if (!$company || $company['status'] !== 'active') {
      View::render('auth/login', ['error'=>'Invalid company details or company inactive.']);
      return;
    }

    $user = User::findByCompanyAndEmail((int)$company['id'], $email);
    if (!$user || $user['status'] !== 'active' || !password_verify($password, $user['password_hash'])) {
      View::render('auth/login', ['error'=>'Invalid login credentials.']);
      return;
    }

    User::updateLastLogin((int)$user['id']);

    Auth::login([
      'id'=>$user['id'],
      'company_id'=>$user['company_id'],
      'name'=>$user['name'],
      'email'=>$user['email'],
      'role'=>$user['role']
    ]);

    Response::redirect('/dashboard');
  }

  public function logout(): void {
    CSRF::verify();
    Auth::logout();
    Response::redirect('/login');
  }
}
